#ifndef __TMatrix__
#define __TMatrix__

//	===========================================================================

#include "../Basics/CCountedObject.hpp"
using Exponent::Basics::CCountedObject;


namespace Exponent
{
	namespace Collections
	{
		/**
		 * @class TMatrix TMatrix.hpp
		 * @brief Template'd 2d array of a fixed size
		 *
		 * @date 16/12/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TMatrix.hpp,v 1.9 2007/02/08 21:06:44 paul Exp $
		 */
		template<class TypeName> class TMatrix : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 * @param d1 Dimension 1 size
			 * @param d2 Dimension 2 size
			 */
			TMatrix(const long d1, const long d2);

			/**
			 * Default construction
			 */
			TMatrix();

			/**
			 * Destruction
			 */
			~TMatrix();

//	===========================================================================

			/**
			 * Assignment operator
			 * @param other The vector to copy
			 * @retval TMatrix& A reference to this
			 */
			TMatrix &operator = (const TMatrix<TypeName> &other);

			/**
			 * Set all elements to
			 * @param element The element to set all to
			 * @retval TMatrixy& A reference to this
			 */
			TMatrix &operator = (const TypeName &element);

			/**
			 * Index operator
			 * @param index The dimension1 index to retrieve
			 */
			TypeName *operator [] (const long index) { return m_pointers[index]; }

//	===========================================================================

			/**
			 * intialise the array with a specific size
			 * @param d1 Dimension 1 size
			 * @param d2 Dimension 2 size
			 */
			void initialise(const long d1, const long d2);

			/**
			 * Set all elements to
			 * @param element The element to set all to
			 */
			void setAllElementsTo(const TypeName &element);

			/**
			 * Get the element at a specific index
			 * @param d1 Dimension 1 index
			 * @param d2 Dimension 2 index
			 * @retval TypeName* The object requested or NULL on error
			 */
			TypeName *elementAtIndex(const long d1, const long d2);

			/**
			 * Insert a pointer at a specific index
			 * @param d1 Dimension 1 index
			 * @param d2 Dimension 2 index
			 * @param object The object to add
			 */
			void addElementAtIndex(const long d1, const long d2, const TypeName &object);

//	===========================================================================

			/**
			 * Is object in array
			 * @param object The object to look for
			 * @retval bool True if this object is stored in the array, false otherwise
			 */
			bool isElementInArray(const TypeName &object);

			/**
			 * Get the size of the array
			 * @retval long The dimension 1 of the array
			 */
			long getDimension1Size() const { return m_dimension1; }

			/**
			 * Get the size of the array
			 * @retval long The dimension 2 of the array
			 */
			long getDimension2Size() const { return m_dimension2; }

			/**
			 * Clear the array - releases all objects
			 */
			void clearArray();

			/**
			 * Get internal buffer
			 * @retval const TypeName ** The internal buffer
			 */
			const TypeName **getInternalBuffer() const { return m_pointers; }

			/**
			 * Get the internal buffer
			 * @retval const TypeName ** The internal buffer
			 */
			TypeName **getMutableInternalBuffer() { return m_pointers; }

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * Free pointers -> deleteds (but not allways) all pointers
			 */
			void freePointers();

//	===========================================================================

			TypeName **m_pointers;				/**< the double array of pointers */
			long m_dimension1;					/**< Dimension 1 size */
			long m_dimension2;					/**< Dimension 2 size */

//	===========================================================================

		};

		/**
		 * @cond
		 */

		EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TMatrix<TypeName>, TypeName, CCountedObject);

//	===========================================================================
		template<class TypeName> TMatrix<TypeName>::TMatrix(const long d1, const long d2) : m_pointers(NULL), m_dimension1(0), m_dimension2(0)
		{
			EXPONENT_CLASS_CONSTRUCTION(TMatrix<TypeName>);
			NULL_POINTER(m_pointers);
			this->initialise(d1, d2);
		}

//	===========================================================================
		template<class TypeName> TMatrix<TypeName>::TMatrix() : m_pointers(NULL), m_dimension1(0), m_dimension2(0)
		{
			EXPONENT_CLASS_CONSTRUCTION(TMatrix<TypeName>);
			NULL_POINTER(m_pointers);
			m_dimension1 = 0;
			m_dimension2 = 0;
		}

//	===========================================================================
		template<class TypeName> TMatrix<TypeName>::~TMatrix()
		{
			EXPONENT_CLASS_DESTRUCTION(TMatrix<TypeName>);
			this->freePointers();
		}

//	===========================================================================
		template<class TypeName> TMatrix<TypeName> &TMatrix<TypeName>::operator = (const TMatrix<TypeName> &other)
		{
			if (&other != this)
			{
				// Clear the old array
				this->freePointers();

				// Copy the sizes
				m_dimension1 = other.m_dimension1;
				m_dimension2 = other.m_dimension2;

				// Create dimension 1
				m_pointers = new TypeName*[m_dimension1];

				// Create dimension 2
				for (long i = 0; i < m_dimension1; i++)
				{
					// Create array
					m_pointers[i] = new TypeName[m_dimension2];

					// Copy it
					memcpy(m_pointers[i], other.m_pointers[i], m_dimension2 * sizeof(TypeName));
				}
			}

			return *this;
		}

//	===========================================================================
		template<class TypeName> TMatrix<TypeName> &TMatrix<TypeName>::operator = (const TypeName &element)
		{
			this->setAllElementsTo(element);
			return *this;
		}

//	===========================================================================
		template<class TypeName> void TMatrix<TypeName>::initialise(const long d1, const long d2)
		{
			// Delete the old array
			this->freePointers();

			// Store the dimensions
			m_dimension1 = d1;
			m_dimension2 = d2;

			// Create dimension 1
			m_pointers = new TypeName*[m_dimension1];

			// Create dimension 2
			for (long i = 0; i < m_dimension1; i++)
			{
				m_pointers[i] = new TypeName[m_dimension2];
			}
		}

//	===========================================================================
		template<class TypeName> void TMatrix<TypeName>::setAllElementsTo(const TypeName &element)
		{
			for (long i = 0; i < m_dimension1; i++)
			{
				for (long j = 0; j < m_dimension2; j++)
				{
					m_pointers[i][j] = element;
				}
			}
		}

//	===========================================================================
		template<class TypeName> TypeName *TMatrix<TypeName>::elementAtIndex(const long d1, const long d2)
		{
			if (m_pointers && (d1 >= 0 && d1 < m_dimension1) && (d2 >= 0 && d2 < m_dimension2))
			{
				return &m_pointers[d1][d2];
			}
			return NULL;
		}

//	===========================================================================
		template<class TypeName> void TMatrix<TypeName>::addElementAtIndex(const long d1, const long d2, const TypeName &object)
		{
			if (m_pointers && (d1 >= 0 && d1 < m_dimension1) && (d2 >= 0 && d2 < m_dimension2))
			{
				m_pointers[d1][d2] = object;
			}
		}

//	===========================================================================
		template<class TypeName> void TMatrix<TypeName>::clearArray()
		{
			this->freePointers();
		}

//	===========================================================================
		template<class TypeName> void TMatrix<TypeName>::freePointers()
		{
			if (m_pointers)
			{
				for (long i = 0; i < m_dimension1; i++)
				{
					FREE_ARRAY_POINTER(m_pointers[i]);
				}
				FREE_ARRAY_POINTER(m_pointers);
				m_dimension1 = 0;
				m_dimension2 = 0;
			}
		}

		/** @endcond */
	}
}
#endif	// End of TMatrix.hpp